#!/usr/bin/env lua5.3

local signal = require 'cqueues.signal'



---
-- Implements common unix daemon signal handling paradigms: exit
-- gracefully on SIGTERM/SIGINT, initiate a configuration reload
-- on SIGHUP.
--
local _M = {}

local callbacks = {}

local stop = false
local reload = false
local last = false
local sl


---
-- Sets a callback for when the signals module changes state.
--
-- @param cb Callback function to call when a signal is
--					received.
--
function _M.set_callback( cb )
	callbacks[ #callbacks + 1 ] = cb
end



---
-- Start the signal listener.
--
-- @param cq The cqueue controller to attach the thread to.
--
function _M.start( cq )

	if not sl then
		sl = signal.listen(
			signal.SIGTERM,
			signal.SIGHUP,
			signal.SIGINT
		)

		signal.block(
			signal.SIGTERM,
			signal.SIGHUP,
			signal.SIGINT
		)
	end

	last = false
	stop = false
	reload = false

	cq:wrap(function()
		repeat

			local sig = sl:wait(1)
			last = sig

			if sig == signal.SIGTERM or sig == signal.SIGINT then
				stop = true
			end

			if sig == signal.SIGHUP then
				stop = true
				reload = true
			end

			if stop or reload then
				for i, callback in ipairs(callbacks) do	-- luacheck: ignore 213
					callback()
				end
			end

		until stop
	end)

end


---
-- Check the last seen signal.
--
-- Mostly used for unit testing.
--
function _M.last_signal()
	return last
end


---
-- Check if the server is still running.
--
-- @return Boolean. False if the server should continue normally;
--			true if the server is beginning to shut down.
--
function _M.is_stopping()
	return stop
end


---
-- Check if the server is preparing to reload itself.
--
-- @return Boolean. False if the server should continue normally;
--			true if the server is preparing to reload it's config.
--			Note that 'true' implies @{is_stopping} is true; however
--			@{is_stopping} could be true while is_reloading is false.
--
function _M.is_reloading()
	return reload
end


---
-- Shuts down the signal listener. Returns immediately;
-- the listener may take a few seconds to terminate.
--
-- This will also call the callbacks. It probably shouldn't though.
--
function _M.stop()
	stop = true
end

return _M
