#!/usr/bin/env lua5.3

local unix = require 'unix'


---
-- A brute-force daemon recycler.
--
-- The daemon calls the exec() system call with it's own arguments,
-- thus reloading back into it's original state. In some cases, this
-- may cause a short service interruption for however long it takes
-- the daemon to renter it's main loop and begin servicing requests.
--
-- While pretty severe, all memory leaks and (usually) file handles
-- are destroyed in the process, as well as any configuration changes
-- reloaded. Lua may be garbage collected, but that doesn't mean the
-- external C libraries it links to are not! Calling exec() generally
-- mitigates most of these concerns for the truly paranoid.
--
-- @module daemonparts.recycle
--
local _M = {}

--
-- What happened writing this module was pretty weird. What started
-- as 30-40 lines that were seriously tricky to get right, when made
-- into a generic function, completely atomized into a small set of
-- trivial looking wrapper functions.
--
-- I've debated scraping this module, but, I know I'll reuse it again
-- and again.
--


local args = {}


---
-- Setup the recycle module. Intended to be called very early,
-- to capture the command line arguments.
--
-- @param cmd The command being called. Usually some variation of
--				/usr/bin/lua or the like. Usually can be determined
--				from argv; however sometimes - such as chain loading -
--				it might not be.
-- @param argv the full arguments array provided to exec().
--
function _M.setup( cmd, argv )

	if cmd then
		args[1] = cmd
	else
		args[1] = argv[0]
	end

	for i, v in ipairs(argv) do -- luacheck: ignore 213
		args[ #args + 1 ] = argv[i]
	end

end



---
-- Recycles the program with exec(). Returns nothing, because the
-- current Lua program ceases to exist.
--
function _M.exec()

	if package.loaded['luacov.runner'] then
		package.loaded['luacov.runner'].save_stats()
	end

	--
	-- Basically impossible to save luacov stats before an exec
	-- syscall.
	--
	-- luacov: disable
	unix.execvp( args[1], args )
	error("Did not complete")
	-- luacov: enable

end


---
-- Recycles the program with exec(). Returns nothing, because the
-- current Lua program ceases to exist. Fork() is called before the
-- execution, causing even the process ID number to be refreshed.
-- This is sometimes, but not always, a helpful thing.
--
function _M.fork_and_exec()

	local pid, err = unix.fork()
	if not pid then
		--
		-- Very difficult to trigger
		--
		-- luacov: disable
		print(err)
		error(err)
		-- luacov: enable
	end

	if pid > 0 then
		-- luacov: disable
		os.exit(0)
		-- luacov: enable
	end

	_M.exec()

end


return _M
