#!/usr/bin/env lua5.3

--local yaml = require "yaml"
local unix = require "unix"

local app_config = {}


---
-- daemonparts.config deals with locating and loading a configuration
-- file from location that, at service startup, is unknown.
--
-- A configurable set of paths are searched for the requested file.
-- Environment variables can be provided to short circuit the search
-- logic for a fixed location.
--
-- A minimal sanity check feature is provided; it can throw errors
-- if a required variable is missing, or if variables are not coercable
-- to the appropriate Lua type.
--
-- The loaded configuration is stored in a singleton software pattern.
-- After configuration is loaded, other contexts within the service
-- can simply call "require 'daemonparts.config'", and then access any
-- configuration variable within the returned table.
--
-- Values are overridable after configuration is loaded, allowing for
-- runtime changes or other possiblities; ie, merging in the results
-- of command line argument parsing.
--
-- @module daemonparts.config
--
local _M = {}


---
-- Defines behavior of the configuration module.
-- @table metaconfig
-- @field name Base name of the service. Defaults to 'daemon'
-- @field required Table of manditory configuration values
-- @field paths Table of paths that are searched, in order, for a valid configuration
--
local mod_config = {
	name = 'daemon',
	required = {},
	paths = {
		'./%s',
		'/usr/local/etc/%s',
		'/etc/%s'
	}
}

---
-- Resets the configuration. All stored values are set to nil;
-- config.setup() will need to be called again to reload all the
-- configuration data afterwards.
--
-- This is mostly intended for unit testing. If you wish to completely
-- reset the state of a running service, check out daemonparts.recycle.
--
-- Takes no arguments and returns nothing.
--
function _M.reset()

	for k in pairs(app_config) do
		app_config[k] = nil
	end

end


---
-- Applies configuration to the configuration module.
--
-- @param vars The configuration values. See @{metaconfig} for definition.
--
function _M.define( vars )

	assert(type(vars) == 'table', 'Invalid config definition')

	for k, v in pairs( vars ) do
		mod_config[k] = v
	end

end


---
-- Locates a configuration file, but does not load it. Includes logic
-- to check for a path specified by an environment variable:
-- if '@{metaconfig}.name_CONFIG' is present, that will be returned.
--
-- @param template Name of configuration file to be found, without
--					directory information. For example, 'config.yml'
--					will search for '/etc/config.yml', etc
-- @return Path of found configuration file.
--
function _M.locate( template )

	local env_name = string.upper(mod_config.name) .. '_CONFIG'
	local ret

	if unix.environ[env_name] then
		return unix.environ[env_name]
	end

	for i, path in ipairs(mod_config.paths) do -- luacheck: ignore 213
		local full_path = string.format(path, template)
		local s = unix.stat(full_path)
		if s and unix.S_ISREG(s.mode) then
			ret = full_path
			break
		end
	end

	if not ret then
		error("Config file " .. tostring(template) .. " not found")
	end

	return ret

end


local load_fcn = dofile

---
-- Sets the function used to load the configuration file. Defaults to
-- dofile(), which means the file should be a valid Lua script that
-- returns a table.
--
-- Using this, the program can call a function that loads YAML, or JSON,
-- or an INI file - whatever you choose. Like dofile(), it should
-- accept one parameter - the file name - and return a table.
--
-- @param fcn The function to actually load and parse the file.
--
function _M.set_loadfunction( fcn )
	load_fcn = fcn
end



---
-- Finds and loads a configuration file. This is the intended entry
-- point of daemonparts.config.
--
-- On success, returns nothing. Throws error if the configuration file
-- isn't found, or is unable to be loaded.
--
-- @param template Name of configuration file to load, without directory.
-- @param path Full path to try. If nil, @{locate} is called.
--
function _M.setup( template, path )

	if not path then
		path = _M.locate( template )
	end

	--local f = assert(io.open(path, "r"))
	--local data = f:read("*all")
	--local x = yaml.load(data)
	--f:close()
	local x = load_fcn( path )

	for k, v in pairs(mod_config.required) do
		assert(x[k] ~= nil, "Manditory config variable missing: " .. k)
		assert(type(x[k]) == v, "Manditory config variable incorrect: " .. k)
	end


	for k, v in pairs(x) do
		rawset(app_config, k, v)
	end

end


setmetatable(app_config, { __index = _M })
return app_config
