# $Id: Session.pm,v 1.1 2001/07/06 05:05:09 fil Exp $
package JAAS::Config::Session;

use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

use POE::Session;

$VERSION = '0.01';
BEGIN {
    if($INC{"JAAS/Object/Services.pm"})
    { *DEBUG=sub () {JAAS::Object::Services->debug()} }
    else { *DEBUG=sub () {0} }
}

###########################################################
sub spawn
{
    my($package, $params)=@_;
    $package||=__PACKAGE__;
    my $heap={};
    $heap->{name}   = $params->{name}||"JAAS_Config";
    $heap->{config} = $params->{config}||JAAS::Config->new();

    POE::Session->create(
        package_states=>[$package=>[qw(_start get get_pb set save load)]],
        heap=>$heap,
    );
}

###########################################################
sub _start
{
    my($kernel, $heap)=@_[KERNEL, HEAP];
    $kernel->alias_set($heap->{name});
}

###########################################################
sub get
{
    my($heap, $path)=@_[HEAP, ARG0];
    DEBUG and warn "Fetching ", join '/', @$path;
    my $rv=$heap->{config}->get(@$path);
    DEBUG and warn "Value is $rv\n";
    return $rv;
}

###########################################################
sub get_pb
{
    my($kernel, $sender, $heap, $path, $postback)=
                            @_[KERNEL, SENDER, HEAP, ARG0, ARG1];

    DEBUG and warn "Fetching ", join('/', @$path), " for ", $sender->ID;
    my $rv=$heap->{config}->get(@$path);
    DEBUG and warn "Value is $rv\n";
    $kernel->post($sender, $postback, $rv);
    DEBUG and warn "Posted to ", $sender->ID, "/$postback";
}

###########################################################
sub set
{
    my($heap, $path)=@_[HEAP, ARG0];
    DEBUG and warn "Setting ", join('/', @{$path}[0..($#$path-1)]), 
                                " to $path->[-1]";
    $heap->{config}->set(@$path);
}

###########################################################
sub load
{
    my($heap, $file)=@_[HEAP, ARG0];
    DEBUG and warn "Loading config file $file";
    $@='';
    $heap->{config}->load($file);
    DEBUG and $@ and warn $@;
}

###########################################################
sub save
{
    DEBUG and warn "Saving config file";
    $@='';
    $_[HEAP]->{config}->save();
    DEBUG and $@ and warn $@;
}

1;
__END__

=head1 NAME

JAAS::Config - JAAS config session

=head1 SYNOPSIS

    use JAAS::Config::Session;

    JAAS::Config::Session->spawn({config=>$config, name=>'Config'});

    my $something=$poe_kernel->call(Config=>'get', \@path);
    $poe_kernel->post(Config=>'get_pb', \@path, 'postback);
    $poe_kernel->post(Config=>'set', [@path, $value]);
    $poe_kernel->post(Config=>'save');
    $poe_kernel->post(Config=>'load', $filename);

=head1 DESCRIPTION

=head1 METHODS

=head2 spawn

=head1 EVENTS

=head2 get

=head2 set

=head2 load

=head2 save

=head1 AUTHOR

Philip Gwyn <perl at pied.nu>

=head1 SEE ALSO

Config::General, perl(1).

=cut


$Log: Session.pm,v $
Revision 1.1  2001/07/06 05:05:09  fil
Added new-fangled DEBUG stuff
Added JAAS::Config::Session (not documented...)

Revision 1.2  2001/07/06 02:11:46  fil
Added documentation to JAAS::Config!

Revision 1.1.1.1  2001/07/06 00:02:58  fil
JAAS config file and object factories.

