# $Id: Config.pm,v 1.3 2001/07/06 05:05:09 fil Exp $
package JAAS::Config;

use strict;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);

use Carp;
use Config::General::Extended;

$VERSION = '0.01';
BEGIN {
    if($INC{"JAAS/Object/Services.pm"})
    { *DEBUG=sub () {JAAS::Object::Services->debug()}; }
    else { *DEBUG=sub () {0}; }
}

###########################################################
sub new
{
    my($package, $filename)=@_;
    my $self=bless {config=>{}, delta=>{}}, $package;
    if($filename and not $self->load($filename)) {
        return;
    }
    return $self;
}

###########################################################
sub load
{
    my($self, $filename)=@_;

    $filename||=$self->{filename};

    my $new=eval {
        croak "No filename to load!" unless $filename;
        my $cge=Config::General::Extended->new(
                        -hash=>{},
                        -UseApacheInclude=>1,
                        # -LowerCaseNames=>1
                        );
        $cge->{configfile}=$filename;
        $cge->_open($filename);
        return $cge->_parse({}, $cge->{content});
    };
    return if $@;
    _merge($self->{config}, $new);
    $self->{filename}||=$filename;
    return 1 unless -r "$filename.delta";

    return $self->load("$filename.delta");
}

###########################################################
sub _merge
{
    my($old, $new)=@_;
    while(my($k, $v)=each %$new) {
        unless(exists $old->{$k}) {
            $old->{$k}=$v;
            next;
        }

        _merge_val($old, $k, $v);
    }
}
sub _merge_val
{
    my($old, $k, $v)=@_;

    my $ov=$old->{$k};
    my $conv=($ov ? (ref($ov)||'SCALAR') : 'UNDEF')."+".
             ($v ? (ref($v)||'SCALAR') : 'UNDEF');
    if($conv eq 'SCALAR+SCALAR') {         # redefine
        $old->{$k}=$v;
    } 
    elsif($conv eq 'SCALAR+HASH') {        # promote, add
        $old->{$k}=$v;
        $old->{$k}{$ov}=1 unless exists $v->{$ov};
    }
    elsif($conv eq 'SCALAR+ARRAY') {       # promote, push
        $old->{$k}=[$ov, @$v];
    }
    elsif($conv eq 'HASH+SCALAR') {        # add
        $ov->{$v}=1;
    }
    elsif($conv eq 'HASH+HASH') {          # merge
        _merge($ov, $v);
    } 
    elsif($conv eq 'HASH+ARRAY') {         # convert to array, push
        $old->{$k}=[$ov, @$v];
        @{$ov}{@$v}=(1) x @$v;
    }
    elsif($conv eq 'ARRAY+SCALAR') {       # push
        push @{$ov}, $v;
    }
    elsif($conv eq 'ARRAY+HASH') {         # push
        push @$ov, $v;
    }
    elsif($conv eq 'ARRAY+ARRAY') {        # push
        push @{$ov}, @$v;
    }
    else {
        carp "Don't know how to merge $conv for '$k'";
        $old->{$k}=$v;
    }
}

###########################################################
sub save
{
    my($self, $ext)=@_;
    $ext||=".delta";
    unless($self->{filename}) {
        $@="No file was loaded, can't save the delta";
        return;
    }
    $@='';
    eval {
        my $cge=Config::General::Extended->new(-hash=>$self->{delta});
        $cge->save($self->{filename}.$ext);
        return 1;
    } or return;
    return 1;
}

###########################################################
sub clear
{
    my($self)=@_;
    $self->{config}={};
    $self->{delta}={};
    delete $self->{filename};
}

###########################################################
sub get
{
    my($self)=@_;
    my $cur=$self->{config};
    foreach my $q (1..($#_-1)) {            # ignore 0th and last one
        $cur=$cur->{$_[$q]};
        return unless $cur;
        if('HASH' ne ref $cur) {
            carp join('/', @_[1..$q]), " is not a hash ref";
            return;
        }
    }
    return $cur->{$_[-1]};
}

###########################################################
sub for
{
    my($self, $name)=@_;
    return $self->{config}{object}{$name};
}

###########################################################
sub set
{
    my($self)=@_;
    return unless @_ > 2;


    my $cur=$self->{config};
    my $delta=$self->{delta};
    foreach my $q (1..($#_-2)) {            # ignore 0th and last two
        $cur=$cur->{$_[$q]};
        return unless $cur;
        if('HASH' ne ref $cur) {
            carp join('/', @_[1..$q]), " is not a hash ref";
            return;
        }
        $delta=$delta->{$_[$q]}||={};
    }

    _merge($delta, {@_[-2, -1]});
    _merge($cur, {@_[-2, -1]});
    return 1;
}

1;
__END__

=head1 NAME

JAAS::Config - JAAS config file loader

=head1 SYNOPSIS

    use JAAS::Config;

    my $config=JAAS::Config->new();
    $config=JAAS::Config->new("filename");
    $config->load("filename");
    $config->clear();

    my $hashref=$config->get("something");
    
    $hashref=$config->for('object name');
    # equiv to 
    $hashref=$config->get("object", "object name");

    $config->set("object", "object name", "args", some=>'thing');

    # save changes to an extra file
    $config->save();

=head1 DESCRIPTION

JAAS configuration files use an Apache-style format, thanks to
Config::General.  The config files will/can contain anything.  

The configuration is changable at run time, via C<set>.  JAAS::Config keeps
tabs of all changes made after a file is loaded.  These changes can then be
saved into a "delta" file via C<save>.  This file will then be reloaded when
the original config is loaded.  While multple files can be loaded, but only
the first one is considered "supreme", as it were, and the delta is
saved/loaded for that file.

Rough overview of file syntax.

    # this is a comment
    key value
    key=value           # 'key' is now a array
    something=multi \
            line \
            value
    <block>
        key=value       # 'key' is not the same key as previously
        <sub-block>
            more data
            goes here
        </sub-block>
    </block>
    # the next two are roughly equivalent, only one of the os is named
    # name1 and the other name2
    <o name1>               
        more values
    </o> 
    <o name2>
       more values
    </o>

    something <<HERE
    You can put a lot of data here
    plus newlines and so on.  And it will work.
    HERE


The preceeding is equivalent to the following perl structure

    {
        'key' => [ 'value', 'value' ],
        'block' => {
            'key' => 'value',
            'sub-block' => {
                'more' => 'data',
                'goes' => 'here'
            }
        },
        'something' => [
            'multi line value',
            'You can put a lot of data here
plus newlines and so on.  And it will work.'
        ],
        'o' => {
            'name1' => { 'more' => 'values' },
            'name2' => { 'more' => 'values' }
        }
    }

Of special importance (now) are <object> blocks </object>.  These are use to
run JAAS::Factory and have a special shortcut, C<for>.  Each object must
have a C<name> after it, so that JAAS::Factory can reference it.

    <object name>
        like so
    </object>

DON'T DO THE FOLLOWING:

    object name

This will cause things to break.

=head1 METHODS

=head2 new

    my $conf=JAAS::Config->new();
    $conf=JAAS::Config->new($filename);

Creates a new JAAS::Config object.  If C<$filename> is specified, it is
loaded and parsed.  If there was an error during parsing, C<new()> returns
undef and $@ is set to whatever error Config::General sees fit to die on.

=head2 load

    $conf->load($filename);

Load a configuration file and merge it's contents with the current
configuration data.  Much magic happens during the merge.  An attempt is
made to make it DWIM without suprises, but there are some gotchas, due
mostly to magic inside of Config::General.

If you DON'T want to merge, but would rather start afresh, call
C<clear()> first.

=head2 save

    $conf->save();
    $conf->save($extention);

Saves any changes to the configuration to a "delta" file.  The file has the
same name as the first loaded file with C<$extention> appended. 
C<$extention> defaults to '.delta'.  If you use a different extention,
C<load()> won't be able to find it, however.  This is here because some
platforms don't like multple extentions... if you know of one, please tell
me how to detect it via C<$^O> and what the default should be.

=head2 clear

    $conf->clear();

Drops everything from the object.  All configuration data, all changes 
is lost.  Clears the "supreme" filename used by C<save()>
    
=head2 get

    my $value=$conf->get(@path);

Returns a single value from the configuration.  C<@path> is all the names
to be looked up, recursively to get to the value.  If one of the names isn't a
hash, well you get a warning and it returns C<undef()>.

    # assuming the config file defined in the DESCRIPTION
    my $here=$conf->('block', 'sub-block', 'goes');
    # $here = 'here'

Note that the returned value can be a scalar, hashref or arrayref.

=head2 set

    $conf->set(@path, $value);

Sets a new value in the configuration.  C<@path> is described previously. 
Some magic is done to merge C<$value> into the destination.  I don't feel
like fully describing this magic here.

=head2 for

    my $c=$conf->for($name);

As mentioned earlier, C<for> is a special accessor used by JAAS::Factory. 
It returns the <object>block</object> named C<$name>.  That is, the
following two are equivalent:

    $c=$conf->for($name);
    $c=$conf->get('object', $name);

See L<JAAS::Factory> for a full description of what should be in the object
configuration.  Here is a small sample.

    <object name>
        class=My::Package  # or package=
        ctor=new           # constructor.  'new' and 'spawn' are default
        <args>
            hashref passed
            to ctor
        </args>
    </object>

=head1 AUTHOR

Philip Gwyn <perl at pied.nu>

=head1 SEE ALSO

Config::General, perl(1).

=cut


$Log: Config.pm,v $
Revision 1.3  2001/07/06 05:05:09  fil
Added new-fangled DEBUG stuff
Added JAAS::Config::Session (not documented...)

Revision 1.2  2001/07/06 02:11:46  fil
Added documentation to JAAS::Config!

Revision 1.1.1.1  2001/07/06 00:02:58  fil
JAAS config file and object factories.

